/*
 * HSSFInterface.java
 *
 * Created on 04 December 2003, 22:29
 */
package ExcelInterface;
/**
 *
 * @author  KURT DE VOS
 */
import ij.*;
import ij.text.*;
import ij.measure.*;
import ij.io.*;
import ij.gui.*;
import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.poifs.filesystem.*;
import java.io.*;
import javax.swing.*;
import IJUtilities.*;


/** This class contains methods to create, load and modify HSSF workbooks
 * @author Kurt De Vos
 * @version 1.0
 */
public class HSSFIO {
    
    private boolean debug = true;
    private FileOutputStream out;
    public HSSFWorkbook wb;
    public HSSFSheet s;
    public String sheetName;
    private HSSFRow r = null;
    private HSSFCell c = null;
    private boolean save;
    private String filePath = null; //Path to Excel File
    private static final String FILE_SEPARATOR = System.getProperty("file.separator"); // system independent file separator
    private static String DEFAULT_PATH = Prefs.getHomeDir()+FILE_SEPARATOR; // IJ dir
    
    /** Creates a new instance of HSSFInterface */
    public HSSFIO() {
    }
    
    /** Use this method to set the filePath when no JChooserDialog is wanted
     * e.g.for automatization and batch processing
     * @param path path to HSSF WorkBook
     */
    public void setFilePath(String path){
        filePath = path;
    } // end of setFilePath(String path)
    
    /** Use this method to retrieve the filePath when no JChooserDialog is wanted
     * e.g.for automatization and batch processing
     * @return String FilePath
     */
    public String getFilePath(){
        return filePath;
    } // end of getFilePath(
    
    /** Method to load an existing Excel file */
    public void loadExcelFile(){
        save = false;
        filePath = getFileName();
        if (filePath == ""){ // no path when input is canceled :: Abort
            IJ.error("You Need to Specify a Valid Path");
            return;
        }
        if (debug)
            System.out.println("loadExcelFile()::filePath = "+filePath);
        
        if (checkFile(filePath)){ // if the file exist...
            try{ // we will try to load it...
                POIFSFileSystem fs = new POIFSFileSystem(new FileInputStream(filePath)); // load file
                wb = new HSSFWorkbook(fs); // load the workbook
            }catch(IOException ioE){
                System.out.println(ioE.getMessage());
                System.out.println("File Not Found, or no Excel File");
            }
        }
        int sheets = wb.getNumberOfSheets();
        
        if (debug)
            System.out.println("sheets: "+sheets);
        
        if (workSheetChoice()){ //if you want a new ws
            sheetName = "Sheet"+(sheets+1);
            s = wb.createSheet(sheetName); // now make a new worksheet
            
        }
        else{
            int wsID = selectWorkSheet(wb);
            try{
                s = wb.getSheetAt(wsID);
                sheetName = wb.getSheetName(wsID);
            }catch(Exception e){
                System.out.println(e.getMessage());
                System.out.println("worksheet Not Found");
            }
        }
        if (debug)
            System.out.println("File loaded");
    } // end of loadExcelFile()
    
    /** Method to select the active HSSF worksheet
     * @return Worksheet identifier
     * @param wb workbook
     */
    int selectWorkSheet(HSSFWorkbook wb){
        String[] wsList = wsList(wb);
        GenericDialog dg = new GenericDialog("Choose WorkSheet");
        dg.addChoice("Work Sheets",  wsList, wsList[0]);
        dg.showDialog();
        if (dg.wasCanceled()) return -1;
        return dg.getNextChoiceIndex();
    } // end of selectWorkSheet(HSSFWorkbook wb)
    
    /** Method to list all worksheets in active WorkBook
     * @return names of worksheets
     * @param wb workbook
     */
    String[] wsList(HSSFWorkbook wb){
        String[] str = new String[wb.getNumberOfSheets()];
        for (int i=0; i<wb.getNumberOfSheets(); i++){
            str[i] = wb.getSheetName(i);
        }
        return str;
    } //end of wsList(HSSFWorkbook wb)
    
    /** Method to create a new HSSF Excelfile with 1 worksheet */
    public void createExcelFile(){
        save = true;
        filePath = getFileName();
        wb = new HSSFWorkbook(); // we create a new file
        s = wb.createSheet("Sheet"+(1)); // now make a new worksheet
        if (debug)
            System.out.println("File Created with 1 worksheet");
    } // end of createExcelFile()
    
    public void closeExcelFile(){
        wb = null;
    }
    /** method to get filename to save file by selection/input in JFileChooser
     * needs XLSFileFilter.class
     * @return fileName
     * in fact replaced by getFileName_classic() which uses the IJ OpenDialog and is recordable
     */
    String getFileName(){
        JFileChooser fc = null;
        try{
            fc = new JFileChooser();
        }
        catch(Throwable e){
            System.out.println(e.getMessage());
            return "";
        }
        fc.setMultiSelectionEnabled(false); // only one file is selectable
        fc.addChoosableFileFilter(new  XlsFileFilter()); // show only ".xls" files retain the option for *.* (for OS9 ??)
        fc.setCurrentDirectory(new File(DEFAULT_PATH)); // directory shown defaults in IJ home directory
        
        int i = 1;
        if (save)
            i = fc.showSaveDialog(IJ.getInstance());
        else
            i = fc.showOpenDialog(IJ.getInstance());// show the JFileChooser with IJ as parent
        if(i != 0){ // if unsuccessfull::abort
            return "";
        }
        String p = fc.getCurrentDirectory().getPath() + FILE_SEPARATOR + fc.getSelectedFile().getName();
        if (!(p.endsWith(".xls") | p.endsWith(".XLS")) ) // check for extension
            p +=".xls"; //add extension if it was not given as input
        DEFAULT_PATH = fc.getCurrentDirectory().getPath();
        return p;
    } // end of getFileName()
    
    /** method to get filename to save file by selection/input in IJ filechooser
     * @return filename
     */
    String getFileName_classic(){
        SaveDialog sd = new SaveDialog("Select...","Result",".xls");
        String p = sd.getDirectory()+FILE_SEPARATOR+sd.getFileName();
        if (!(p.endsWith(".xls") | p.endsWith(".XLS")) ) // check for extension
            p +=".xls"; //add extension if it was not given as input
        DEFAULT_PATH = sd.getDirectory();
        return p;
    } // end of getFileName_classic()
    
    /** Checks if the file exists
     * @return returns true if file exists
     * @param p file path
     */
    boolean checkFile(String p){ // does the file exist ?
        return new File(p).isFile();
    } // end of checkFile(String p)
    
    /** Asks if a new worksheet is wanted
     * Defaults false
     * @return true if an new worksheet is wanted
     */
    boolean workSheetChoice(){
        YesNoCancelDialog YNCdg = new YesNoCancelDialog(IJ.getInstance(), " New Worksheet" , "Do you want to create a new worksheet?");
        if (YNCdg.yesPressed()) return true;
        return false;
    } // end of workSheetChoice()
    
    /** Adds a worksheet to a workbook */
    public void addWorkSheet(){
        try{
            sheetName = "IJResults"+(wb.getNumberOfSheets()+1);
            s = wb.createSheet(sheetName);
        }catch(Exception e){
            System.out.println(e.getMessage());
        }
        if (debug)
            System.out.println("WorkSheet Added");
    } // end of addWorkSheet()
    
    /** Deletes a worksheet from a workbook */
    public void deleteWorkSheet(){
        int wsID = selectWorkSheet(wb);
        try{
            wb.removeSheetAt(wsID);
        }catch(Exception e){
            System.out.println(e.getMessage());
        }
        if (debug)
            System.out.println("WorkSheet Deleted");
    } // end of deleteWorkSheet()
    
    /** Renames a worksheet */
    public void renameSheet(){
        int wsID = selectWorkSheet(wb);
        String name = IJ.getString("New Name...", "");
        try{
            sheetName = name;
            wb.setSheetName(wsID, name);
        }catch(Exception e){
            System.out.println(e.getMessage());
        }
        if (debug)
            System.out.println("WorkSheet Renamed");
    } // end of renameSheet()
    
    public int getActiveSheetIndex(){
        return wb.getSheetIndex(sheetName);
    }
    
    /** slects a worksheet */
    public void selectActiveSheet(){
        int wsID = selectWorkSheet(wb);
        try{
            s = wb.getSheetAt(wsID);
            sheetName = wb.getSheetName(wsID);
        }catch(Exception e){
            System.out.println(e.getMessage());
        }
        if (debug)
            System.out.println("WorkSheet Selected");
    } // end of selectActiveSheet()
    
    /** Saves the Excel file */
    public void saveExcel(){
        try{
            out = new FileOutputStream(filePath);
            wb.write(out);
            System.out.println("Saved: "+ filePath);
            out.close();
        }catch(IOException ioE){
            System.out.println("filepath exception "+ filePath);
            System.out.println(ioE.getMessage());
            IJ.error("You Need to Specify a Valid Path \n Save not Possible"); // should not happen
        }
        if (debug)
            System.out.println("File saved");
    } // end of saveExcel()
    
} // end of HSSFIO.class
