/*
 * HSSFUtils.java
 *
 * Created on 04 December 2003, 22:33
 */

package ExcelInterface;

/**
 *
 * @author  KURT DE VOS
 */
import ij.*;
import ij.measure.*;
import ij.text.*;
import ij.gui.*;

import org.apache.poi.hssf.usermodel.*;
import org.apache.poi.poifs.filesystem.*;

import java.io.*;

//import javax.swing.*;

import IJUtilities.*;
import SystemUtils.*;

/** This class contains methods to write data to a HSSF workbook */
public class HSSFWriteUtils {
    /** Used for debugging */
    boolean debug = true;
    HSSFWorkbook wb;
    HSSFSheet s;
    HSSFRow r;
    HSSFCell c;
    
    /** Creates a new instance of HSSFUtils */
    public HSSFWriteUtils() {
    }
    
    /** sets the active workbook
     * @param HSSFwb workbook
     */
    public void setWb(HSSFWorkbook HSSFwb){
        this.wb = HSSFwb;
    }
    
    /** sets the active worksheet
     * @param HSSFs worksheet
     */
    public void setSheet(HSSFSheet HSSFs){
        this.s = HSSFs;
    }
    
    /** Style for Headers - Bold, Centered
     * @return HSSFCellStyle
     */
    HSSFCellStyle headingCellStyle(){
        HSSFFont font = wb.createFont(); // defaults to Arial on windows
        font.setBoldweight(HSSFFont.BOLDWEIGHT_BOLD); // bold
        HSSFCellStyle style = wb.createCellStyle(); // make style
        style.setFont(font); // set font
        style.setAlignment(HSSFCellStyle.ALIGN_CENTER); // set alignment to centered
        return style;
    } // headingCellStyle()
    
    /** Write the data to Excel as row
     * @param data String data to be written
     */
    public void writeDataAsRow(String data){
        int row = s.getLastRowNum();
        r = s.createRow((short)(row+1));
        r.setRowNum((short)(row+1));
        
        String[] dataTextArray = new StringUtils().split(data); // parse the data to an array
        for (short cellnum = (short) 0; cellnum < dataTextArray.length; cellnum++){
            c = r.createCell((short)(r.getLastCellNum()+1));
            try{
                float value = Float.parseFloat(dataTextArray[cellnum]); // try to convert string to float
                c.setCellValue(value); // if it is a float write it
            } catch(Exception e){
                System.out.println("Float error " +e.getMessage());
                c.setCellValue(dataTextArray[cellnum]); // in case it is no number the try{} jumps here and writes the non-Float
            }
        }
        if (debug)
            System.out.println("Row written");
    } //  writeDataAsRow(String data)
    
    /** Write the data to Excel as column
     * @param data Float[] data from ResultsTable to be written
     */
    public void writeDataAsColumn(float[] data){
        for (int i=1; i<=data.length+1; i++){ // if the row doesn't exist make it +1 leaves room for Column Headings
            if (s.getRow(i)==null){
                r = s.createRow((short)(i));
                r.setRowNum((short)(i));
            }
        }
        for (short rownum = (short) 0; rownum < data.length; rownum++){
            r = s.getRow(rownum+1);
            c = r.createCell((short)(r.getLastCellNum()+1));
            try{
                float value = data[rownum]; // try to convert string to float
                c.setCellValue(value); // if it is a float write it
            } catch(Exception e){
                System.out.println("Float error " +e.getMessage());
                c.setCellValue(data[rownum]); // in case it is no number the try{} jumps here and writes the non-Float
            }
        }
        if (debug)
            System.out.println("Column written");
    } //  writeDataAsColumn(String data)
    
    /** Selects to column to be written based on column heading
     * @param rt ij.measure.ResultsTable.getResultsTable()
     * @return Column Header
     */
    String selectColumnHeading(ResultsTable rt){
        String[] columnHeadings = new StringUtils().split(rt.getColumnHeadings());
        GenericDialog dg = new GenericDialog("Select Column...");
        dg.addChoice("Columns",  columnHeadings, columnHeadings[0]);
        dg.showDialog();
        if (dg.wasCanceled())
            return null;
        return columnHeadings[dg.getNextChoiceIndex()];
    }
    
    /** Write all column headings to Excel
     * @param headings Column Headings (ResultsTable.getColumnHeadings())
     */
    public void writeColumnHeadings(String headings){
        String[] headArray = new StringUtils().split(headings); // parse the headings to an array
        if (!isRT){ // to allow for workaround for first blank column heading
            if (s.getRow(0) == null)
                r = s.createRow((short)(0)); // create a row to contain the headings
            else
                r = s.createRow((short)(s.getLastRowNum()+2));
        }
        // @todo if there is already a table on the sheet, append not overwrite
        for (short cellnum = (short) 0; cellnum < headArray.length; cellnum++){
            try{
                c = r.createCell((short)(r.getLastCellNum()+1));
                c.setCellValue(headArray[cellnum]);
                c.setCellStyle(headingCellStyle());
            }catch(Exception e){
                System.out.println(e.getMessage());
            }
        }
        if (debug)
            System.out.println("Column Headings Written");
    } // writeHeadings(String headings)
    
    /** Write one column header to Excel
     * @param heading column header of column to be written
     */
    public void writeOneColumnHeading(String heading){
        if (s.getRow(0) == null)
            r = s.createRow((short)(0)); // create a row to contain the headings
        else
            r=s.getRow(0);
        try{
            c = r.createCell((short)(r.getLastCellNum()+1));
            c.setCellValue(heading);
            c.setCellStyle(headingCellStyle());
        }catch(Exception e){
            System.out.println(e.getMessage());
        }
        if (debug)
            System.out.println("One Column Heading Written");
    } // writeOneColumnHeading(String heading)
    
    /** Checks if a ij.measure.ResultsTable.getResultsTable() is being used */
    boolean isRT = false;
    
    /** Write IJ ResultsTable to Excel
     * @param rt ij.measure.ResultsTable.getResultsTable()
     */
    public void writeRsTable(ResultsTable rt){
        isRT = true;
        if (s.getRow(0) == null)
            r = s.createRow((short)(0)); // create a row to contain the headings
        else
            r = s.createRow((short)(s.getLastRowNum()+2));
        c = r.createCell((short)(r.getLastCellNum()+1)); // workaround for first blank column heading
        c.setCellValue("Count");
        c.setCellStyle(headingCellStyle());
        if (debug){
            System.out.println("Count written");
        }
        /* Write the Column Headers */
        String headings = rt.getColumnHeadings(); // get the column headings
        if (debug){
            System.out.println("headings: "+headings.toString());
        }
        writeColumnHeadings(headings);
        /* Write the Data */
        for (short rownum = (short) 0; rownum < rt.getCounter(); rownum++){
            String data = rt.getRowAsString(rownum); // get the data
            if (debug){
                System.out.println("data: row:"+rownum+" data: "+data.toString());
            }
            writeDataAsRow(data);
        }
        if (debug){
            System.out.println("ResultsTable Written");
        }
        isRT = false;
    } // writeResultsTableToExcel(ResultsTable rt)
    
    /** Method to write the contents (tabular) of any ij.text.TextPanel to Excel
     * @param tp TextPanel
     */
    public void writeTextPanel(TextPanel tp){
        /* Write the Column Headers */
        String headings = tp.getColumnHeadings(); // get the column headings
        writeColumnHeadings(headings);
        /* Write the Data */
        for (short rownum = (short) 0; rownum < tp.getLineCount(); rownum++){
            String data = tp.getLine(rownum); // get the data
            writeDataAsRow(data);
        }
        if (debug)
            System.out.println("TextPanel Headings Written");
    } //  writeTextPanelToExcel(TextPanel tp)
    
    /** Write last row of ResultsTable to Excel
     * @param data ResultsTable.getLastRow()
     */
    public void appendLastLine(String data){
        if (wb != null && s != null){
            writeDataAsRow(data);
        }
        else{
            IJ.error("You need an Excel File");
        }
        if (debug)
            System.out.println("Row Appended");
    }
    
    /** Deletes last row of worksheet
     * Use as undo
     */
    public void deleteLastRow(){
        if (wb != null && s != null){
            s.removeRow(s.getRow(s.getLastRowNum()));
        }
        else{
            IJ.error("You need an Excel File");
        }
        if (debug)
            System.out.println("Row Deleted");
    }
    
    /** Write one column of ResultsTable to Excel
     * @param rt ij.measure.ResultsTable.getResultsTable()
     */
    public void writeOneColumn(ResultsTable rt){
        if (wb != null && s != null){
            String column = selectColumnHeading(rt);
            writeOneColumnHeading(column);
            writeDataAsColumn(rt.getColumn(rt.getColumnIndex(column)));
        }
        else{
            IJ.error("You need an Excel File");
        }
        if (debug)
            System.out.println("Column Written");
    }
    
    /** Write one column of worksheet
     * Column is selected based on ResultsTable header
     * @param rt ij.measure.ResultsTable.getResultsTable()
     */
    public void deleteOneColumn(ResultsTable rt){
        short colID = 0;
        if (wb != null && s != null){
            String column = selectColumnHeading(rt);
            r = s.getRow(0);
            for (short j=0; j<=r.getLastCellNum(); j++){
                c = r.getCell(j);
                if (c.getStringCellValue().equalsIgnoreCase(column)){
                    colID = c.getCellNum();
                    for (int i=0; i<=s.getLastRowNum(); i++){
                        r = s.getRow(i);
                        r.removeCell(r.getCell(colID));
                    }
                }
            }
        }
        else{
            IJ.error("You need an Excel File");
        }
        if (debug)
            System.out.println("Column Deleted");
    }
    
    /** Write clipboard to Excel */
    public void writeFromClipboard(){
        String data = ClipHelper.pasteString();
        writeDataAsRow(data);
        if (debug)
            System.out.println("ClipBoard Written");
    }
    
}
